/* 
Gestion d'un PAD BLE dans une interface de centrale DCC  
-------------------------------------------------------
Version 1.4 - 2023/08

Fonctionnalités : 
- reconnaissance de PAD modèle R1
- enregistrement (EEPROM) de l'adresse MAC d'un PAD par apprentissage
- oubli d'une adresse MAC enregistrée
- connexion au PAD mémorisé
- interprétation des commandes reçues
- analyse et transfert des trames JMRI vers la centrale
- transfert des trames de la centrale vers JMRI
- mémorisation de la liste des locos commandées
- sélection et pilotage d'une loco mémorisée via le PAD (vitesse, sens, fonctions) 
- enregistrement (EEPROM) de la liste des locos

Note : lors de la connexion du PAD avec l'ESP32, le changement du mode de fonctionnement (A vers C) est obligatoire
       afin de disposer de toutes les fonctionnalités proposées.
       Après la synchronisation, appuyez sur les boutons M + C pour changer le mode de fonctionnement du PAD.
       
NitraThor : https://www.nitrathor.fr
*/

/* librairies */
#include <Wire.h>                                      // bus I2C
#include <ArduinoBLE.h>                                // Bluetooth Low Energy
#include "SSD1306.h"                                   // écran OLED SSD1306
#include <Preferences.h>                               // gestion EEPROM
#include <HardwareSerial.h>                            // gestion ports série ESP32

/* définition des valeurs du pad */
#define BT_LEFT     0x100
#define BT_RIGHT    0x200
#define BT_UP       0x400
#define BT_DOWN     0x800
#define MAC_LEN     20                                 // nombre d'octets de l'adresse MAC
#define RX2         16                                 // pin RX serial 2 (réception de la centrale DCC)
#define TX2         17                                 // pin TX serial 2 (envoi à la centrale DCC)
#define TITRE_MENU "PAD BLE DCC"
#define MENU_RECH  "recherche en cours"
#define MENU_TROU  "PAD trouvé. Mémoriser ?"
#define MENU_CONF  "confirmation ?"
#define MENU_NDEF  "non définie"
#define MENU_NSEL  "aucune loco sélectionnée"
#define TAILLE_BUF  80

#define MASQ_00_04  B10000000
#define MASQ_05_08  B10100000
#define MASQ_09_12  B10110000
#define MASQ_13_20  B11011110
#define MASQ_21_28  B11011111

/* afficheur OLED I2C */
SSD1306  ecran(0x3c, 21, 22);
/* gestion de l'espace EEPROM */
Preferences eeprom;
/* gestion des ports série */
HardwareSerial ESP32Serial(0);
HardwareSerial ESP32Serial2(2);

/* structures */
typedef struct strucLoco strucLoco;
struct strucLoco{                                      // <t registre numéro vitesse direction>
  byte         byReg;                                  // numéro de registre utilisé
  unsigned int iNum;                                   // adresse DCC de la loco
  byte         byVit;                                  // vitesse de 0 à 126
  boolean      boDir;                                  // vrai = marche avant
  boolean      tboFct[29];                             // vrai = fonctions actives
  byte         bySel_fct;                              // numéro de fonction sélectionée
  strucLoco    *ps;                                    // ptr vers la structure suivante
};

/* constantes */
                const byte    MAX_MENU                 =  3;
                const byte    MAX_MENU_LOCO            =  2;
                const byte    NB_PTS_ATT               =  4; // nombre de points à afficher pour les phases d'attente
                const int     OFFSET_Y                 = 10; // offset de ligne
                const String  MENU[MAX_MENU]           = {"Attente connexion PAD", "Oublier le PAD", "Détection nouveau PAD"};
                const String  MENU_LOCO[MAX_MENU_LOCO] = {"Liste locos détectées", "Locos mémorisées"};
/* variables */
                char          tcMac[MAC_LEN];          // adresse MAC mémorisée
                byte          bySel_menu;              // rubrique sélectionnée du menu principal
                boolean       boSel_menu_ok;           // vrai = rubrique menu principal active (bySel_menu)
                byte          bySel_attente;           // compteur du nombre de points affichés lors d'une attente
                byte          bySel_menu_loco;         // rubrique sélectionnée du menu des locomotives
                int           iPos_x, iPos_y;          // position x et y sur l'écran
                char          tcAdr_ble[64];
                char          tcNom_ble[64];
                String        sMessage;                // contenu du message à afficher (PAD non connecté)
                boolean       boModif;                 // vrai = action réalisée sur le PAD (PAD connecté)
volatile static boolean       boBt_sui;                // vrai = interruption appui bouton suivant
volatile static boolean       boBt_ent;                // vrai = interruption appui bouton entrée
volatile static boolean       boBt_mem;                // vrai = interruption appui bouton mémorisation
volatile static unsigned long lOld_millis_sui          = 0; // délai répétition bouton suivant
volatile static unsigned long lOld_millis_ent          = 0; // délai répétition bouton entrée
volatile static unsigned long lOld_millis_mem          = 0; // délai répétition bouton mémorisation
                unsigned long lOld_millis_att          = 0; // délai répétition points attente (...)
                unsigned long lOld_millis_PAD          = 0; // délai répétition joystick PAD 
                unsigned long lOld_millis_FCT          = 0; // délai répétition défilement fonctions 
                BLEDevice     peripherique;
         static uint16_t      iAction_PAD;             // représentation des actions du PAD
                byte          tbyData[TAILLE_BUF];
                boolean       boDeb_cmd                = false;
                char          tcRec_cmd[TAILLE_BUF];   // réception de commande en cours
                char          tcAff_cmd[TAILLE_BUF];   // affichage de commande en cours
                strucLoco     *p_loco;                 // ptr racine de la liste des locos  
                strucLoco     *p_sel_loco;             // ptr sur sélection liste loco  
                strucLoco     *p_selA_loco;            // ptr sur loco A sélectionnée  
                strucLoco     *p_selB_loco;            // ptr sur loco B sélectionnée  
                strucLoco     *p_selC_loco;            // ptr sur loco C sélectionnée  
                strucLoco     *p_selD_loco;            // ptr sur loco D sélectionnée  
                byte          bySel_memo;              // numéro de loco sélectionnée (0=A, 1=B, 2=C, 3=D)
                boolean       boGen_dcc                = false; // vrai = génération DCC en cours

/* interruption appui bouton suivant */
void IRAM_ATTR bouton_suivant_ISR() {
  if(millis() - lOld_millis_sui >= 250){
    lOld_millis_sui = millis();
    boBt_sui = true;
  }
}

/* interruption appui bouton enter */
void IRAM_ATTR bouton_enter_ISR() {
  if(millis() - lOld_millis_ent >= 250){
    lOld_millis_ent = millis();
    boBt_ent = true;
  }
}

/* scan des périphériques à portée du serveur BLE */
boolean Lecture_BLE(){
char    tcAddr[64];
char    tcName[64];
boolean retour;

  retour = false;
  peripherique = BLE.available();
  if (peripherique){
    sprintf(tcAddr, "%s", peripherique.address().c_str());
    sprintf(tcName, "%s", peripherique.localName().c_str());
    if (String(tcName).indexOf("R1")>-1){
      sprintf(tcAdr_ble, "%s", tcAddr);
      sprintf(tcNom_ble, "%s", tcName);
      retour = true;
    }
  }
  return retour;
}

/* initialisation du scan des périphériques BLE */
void Init_scan_BLE(){
  
  if (String(tcAdr_ble) != ""){
    sprintf(tcAdr_ble, "%s", "");
    sprintf(tcNom_ble, "%s", "");
  }
  BLE.scan();
}

/* récupération adresse MAC stockée sur EEPROM */
void Charge_MAC_EEPROM(){

  eeprom.getString("MAC_ADDR", tcMac, MAC_LEN);
}

/* récupération adresse locos stockée sur EEPROM */
void Charge_Locos_EEPROM(){
strucLoco    *pc;                                          // pointeurs courant sur structure loco
strucLoco    *pp;                                          // pointeurs précédent sur structure loco
unsigned int uiAddr;

  eeprom.begin("LOCO_BLE", false);                         // ouverture de l'espace eeprom LOCO
  uiAddr=eeprom.getUInt("ADDR_A", 0);
  if (uiAddr!=0){
    pc=p_loco;                                             // recherche loco
    while ((pc!=NULL) && (pc->iNum!=uiAddr)) pc=pc->ps;
    if (pc==NULL){                                         // loco non trouvée = à ajouter
      strucLoco *pn=new strucLoco;
      pn->byReg=1;
      pn->iNum=uiAddr;
      pn->byVit=0;
      pn->boDir=true;
      for(int i=0;i<29;i++)pn->tboFct[i]=false;   
      pn->bySel_fct=0;            
      pn->ps=p_loco;      
      p_loco=pn;
      if (p_sel_loco==NULL) p_sel_loco=p_loco;             // init sélection affichage liste des locos
      bySel_menu_loco=1; // emplacement loco mémorisée
      bySel_memo=0;
      p_selA_loco=p_loco;
    }
  }
  uiAddr=eeprom.getUInt("ADDR_B", 0);
  if (uiAddr!=0){
    pc=p_loco;                                             // recherche loco
    while ((pc!=NULL) && (pc->iNum!=uiAddr)) pc=pc->ps;
    if (pc==NULL){                                         // loco non trouvée = à ajouter
      strucLoco *pn=new strucLoco;
      pn->byReg=1;
      pn->iNum=uiAddr;
      pn->byVit=0;
      pn->boDir=true;
      for(int i=0;i<29;i++)pn->tboFct[i]=false;   
      pn->bySel_fct=0;            
      pn->ps=p_loco;      
      p_loco=pn;
      if (p_sel_loco==NULL) p_sel_loco=p_loco;             // init sélection affichage liste des locos
      bySel_menu_loco=1; // emplacement loco mémorisée
      bySel_memo=0;
      p_selB_loco=p_loco;
    }
  }
  uiAddr=eeprom.getUInt("ADDR_C", 0);
  if (uiAddr!=0){
    pc=p_loco;                                             // recherche loco
    while ((pc!=NULL) && (pc->iNum!=uiAddr)) pc=pc->ps;
    if (pc==NULL){                                         // loco non trouvée = à ajouter
      strucLoco *pn=new strucLoco;
      pn->byReg=1;
      pn->iNum=uiAddr;
      pn->byVit=0;
      pn->boDir=true;
      for(int i=0;i<29;i++)pn->tboFct[i]=false;   
      pn->bySel_fct=0;            
      pn->ps=p_loco;      
      p_loco=pn;
      if (p_sel_loco==NULL) p_sel_loco=p_loco;             // init sélection affichage liste des locos
      bySel_menu_loco=1; // emplacement loco mémorisée
      bySel_memo=0;
      p_selC_loco=p_loco;
    }
  }
  uiAddr=eeprom.getUInt("ADDR_D", 0);
  if (uiAddr!=0){
    pc=p_loco;                                             // recherche loco
    while ((pc!=NULL) && (pc->iNum!=uiAddr)) pc=pc->ps;
    if (pc==NULL){                                         // loco non trouvée = à ajouter
      strucLoco *pn=new strucLoco;
      pn->byReg=1;
      pn->iNum=uiAddr;
      pn->byVit=0;
      pn->boDir=true;
      for(int i=0;i<29;i++)pn->tboFct[i]=false;   
      pn->bySel_fct=0;            
      pn->ps=p_loco;      
      p_loco=pn;
      if (p_sel_loco==NULL) p_sel_loco=p_loco;             // init sélection affichage liste des locos
      bySel_menu_loco=1; // emplacement loco mémorisée
      bySel_memo=0;
      p_selD_loco=p_loco;
    }
  }
  eeprom.end();
}

/* réception série depuis le PC */
int Reception_Serie(byte *bMsg, int iLen){
int iBytes = 0;

  while (ESP32Serial.available() > 0){
    bMsg[iBytes++] = (byte)ESP32Serial.read();
    if (iBytes>iLen) break;
  }
  return iBytes;
}

/* transmission série vers le PC */
void Transmission_Serie(byte *bMsg, int iLen){

  ESP32Serial.write(bMsg, iLen);
}

/* réception série depuis la centrale DCC */
int Reception_Serie2(byte *bMsg, int iLen){
int iBytes = 0;

  while (ESP32Serial2.available() > 0){
    bMsg[iBytes++] = (byte)ESP32Serial2.read();
    if (iBytes>iLen) break;
  }
  return iBytes;
}

/* transmission série vers la centrale DCC */
void Transmission_Serie2(byte *bMsg, int iLen){

  ESP32Serial2.write(bMsg, iLen);                 // envoi commande à la centrale
  Reception_caracteres(bMsg, iLen);               // interception commande pour analyse
}

/* affichage du menu principal BLE */
void Affiche_menu_BLE(){
int i;
String pts, aff;

  ecran.clear();
  // affichage titre
  iPos_y = 0;
  iPos_x = (ecran.getWidth()-ecran.getStringWidth(TITRE_MENU))/2;
  ecran.drawString(iPos_x, iPos_y, TITRE_MENU);
  iPos_y += OFFSET_Y;
  iPos_x = 0;
  // affichage titre menu
  if (boSel_menu_ok){
    ecran.drawString(iPos_x, iPos_y, MENU[bySel_menu]);
  }else{
    ecran.drawString(iPos_x, iPos_y, ">"+MENU[bySel_menu]+" ?");
  }
  iPos_y += OFFSET_Y;
  iPos_x = 10;
  // affichage sous-menu
  if (boSel_menu_ok){                                      
    if (bySel_menu == 0){                                   // Attente connexion PAD   
      if(millis() - lOld_millis_att >= 250){
        lOld_millis_att = millis();
        bySel_attente+=1;
        if (!(bySel_attente < NB_PTS_ATT)) bySel_attente = 0;
      }
      pts = "";
      for (i=0;i<bySel_attente;i++) pts+=".";
      aff = String("> ")+MENU_RECH+pts;
      ecran.drawString(iPos_x, iPos_y, aff);      
    }else if (bySel_menu == 1){                            // Oublier PAD
      ecran.drawString(iPos_x, iPos_y, String("> ")+MENU_CONF);
    }else {                                                // Recherche PAD
      if (String(tcAdr_ble) == ""){                        // pas de PAD trouvé
        // affichage recherche en cours
        if(millis() - lOld_millis_att >= 250){
          lOld_millis_att = millis();
          bySel_attente+=1;
          if (!(bySel_attente < NB_PTS_ATT)) bySel_attente = 0;
        }
        pts = "";
        for (i=0;i<bySel_attente;i++) pts+=".";
        ecran.drawString(iPos_x, iPos_y, String("> ")+MENU_RECH+pts);      
        Lecture_BLE();
      }else{                                               // PAD trouvé
        BLE.stopScan();
        ecran.drawString(iPos_x, iPos_y, MENU_TROU);
        iPos_y += OFFSET_Y;
        ecran.drawString(iPos_x, iPos_y, String(tcNom_ble)); 
        iPos_y += OFFSET_Y;
        ecran.drawString(iPos_x, iPos_y, String(tcAdr_ble)); 
      }
    }
  }
  if (sMessage!=""){
    iPos_y += 2*OFFSET_Y;
    iPos_x = (ecran.getWidth()-ecran.getStringWidth(sMessage))/2;
    ecran.drawString(iPos_x, iPos_y, sMessage); 
  }
  ecran.display();
}

/* gestion de tous les cas du menu principal */
void Action_menu_BLE(){
int i;

  sMessage = "";
  /* gestion bouton suivant */
  if (boBt_sui){                           // appui sur bouton suivant
    if (!boSel_menu_ok){                   // navigation menu principal
      bySel_menu+=1;
      if (!(bySel_menu < MAX_MENU)) bySel_menu = 0;
    }else{                                 // sous-menu actif
      if (bySel_menu == 0){                 // sous-menu actif : attente connexion PAD
        BLE.stopScan();                    // arrêt scan BLE
      }else if (bySel_menu == 1){           // sous-menu actif : oublier PAD
        /* aucune action */
      }else {                              // sous-menu actif : nouveau PAD
        BLE.stopScan();                    // arrêt scan BLE
        if (String(tcAdr_ble) != ""){
          sprintf(tcAdr_ble, "%s", "");
          sprintf(tcNom_ble, "%s", "");
        }
      }
      boSel_menu_ok = false;               // retour au menu principal
    }
    boBt_sui = false;
  } 
  /* gestion bouton enter */
  if (boBt_ent){                            // appui sur bouton enter
    if (boSel_menu_ok){                     // le sous-menu est déjà actif  
      if (bySel_menu == 0){                 // sous-menu actif : attente connexion PAD
        /* aucune action */
      }else if (bySel_menu == 1){           // sous-menu actif : oublier PAD      
        /* validation oubli adresse MAC */
        eeprom.remove("MAC_ADDR");
        sprintf(tcMac, "%s", "");
        sMessage = "Adresse PAD oubliée";
        boSel_menu_ok = false;              // retour au menu principal
      }else{                                // sous-menu actif : nouveau PAD
        if (String(tcAdr_ble) != ""){       // nouveau PAD trouvé
          BLE.stopScan();
          /* enregistrement nouvelle adresse MAC */
          eeprom.putString("MAC_ADDR", String(tcAdr_ble));
          Charge_MAC_EEPROM();
          sMessage = "Nouveau PAD mémorisé";
          boSel_menu_ok = false;            // retour au menu principal
        }
      }
    }else{                                  // validation sélection item du menu principal
      if (bySel_menu == 0){                 // validation sélection attente connexion PAD
        bySel_attente = 0;
        BLE.scan();
      }else if (bySel_menu == 1){           // validation sélection oublier PAD     
        /* aucune action */
      }else {                               // validation sélection nouveau PAD
        Init_scan_BLE();
      }
      boSel_menu_ok = true;                 // sous-menu actif maintenant
    }
    boBt_ent = false;                       // réinitialisation appui bouton enter
  }
}

/* détection d'un évènement PAD */
void HIDReportWritten(BLEDevice central, BLECharacteristic characteristic){
int iLen, i;
uint8_t tcTemp[128];

  iLen = characteristic.readValue(tcTemp, sizeof(tcTemp)); // lecture des informations du PAD
  Traitement_actions_PAD(tcTemp, iLen);                    // analyse des actions réalisées sur le PAD
}

/* envoyer la commande de la loco sur le port série */
void Envoyer_commande_loco(strucLoco *pc){
char tcCde_pad[TAILLE_BUF];
byte tbCde_pad[TAILLE_BUF];

  if (pc->boDir)sprintf(tcCde_pad, "<t %d %d %d 1>", pc->byReg, pc->iNum, pc->byVit);
  else sprintf(tcCde_pad, "<t %d %d %d 0>", pc->byReg, pc->iNum, pc->byVit);
  memcpy(tbCde_pad,tcCde_pad,strlen(tcCde_pad));
  Transmission_Serie2(&tbCde_pad[0], strlen(tcCde_pad));
}

/* envoyer la commande des fonctions de la loco sur le port série */
void Envoyer_commande_fonctions(strucLoco *pc, int iModif){
char    tcCde_pad[TAILLE_BUF];
byte    tbCde_pad[TAILLE_BUF];
byte    byOctet1, byOctet2;

  if (iModif<5){
    byOctet1=MASQ_00_04;
    if (pc->tboFct[0]) byOctet1|=B00010000;
    if (pc->tboFct[1]) byOctet1|=B00000001;
    if (pc->tboFct[2]) byOctet1|=B00000010;
    if (pc->tboFct[3]) byOctet1|=B00000100;
    if (pc->tboFct[4]) byOctet1|=B00001000;
    sprintf(tcCde_pad, "<f %d %d>", pc->iNum, byOctet1);
    memcpy(tbCde_pad,tcCde_pad,strlen(tcCde_pad));
    Transmission_Serie2(&tbCde_pad[0], strlen(tcCde_pad));
  }else if (iModif<9){
    byOctet1=MASQ_05_08;
    if (pc->tboFct[5]) byOctet1|=B00000001;
    if (pc->tboFct[6]) byOctet1|=B00000010;
    if (pc->tboFct[7]) byOctet1|=B00000100;
    if (pc->tboFct[8]) byOctet1|=B00001000;
    sprintf(tcCde_pad, "<f %d %d>", pc->iNum, byOctet1);
    memcpy(tbCde_pad,tcCde_pad,strlen(tcCde_pad));
    Transmission_Serie2(&tbCde_pad[0], strlen(tcCde_pad));
  }else if (iModif<13){
    byOctet1=MASQ_09_12;
    if (pc->tboFct[9])  byOctet1|=B00000001;
    if (pc->tboFct[10]) byOctet1|=B00000010;
    if (pc->tboFct[11]) byOctet1|=B00000100;
    if (pc->tboFct[12]) byOctet1|=B00001000;
    sprintf(tcCde_pad, "<f %d %d>", pc->iNum, byOctet1);
    memcpy(tbCde_pad,tcCde_pad,strlen(tcCde_pad));
    Transmission_Serie2(&tbCde_pad[0], strlen(tcCde_pad));
  }else if (iModif<21){
    byOctet1=MASQ_13_20;
    byOctet2=B00000000;
    if (pc->tboFct[13]) byOctet2|=B00000001;
    if (pc->tboFct[14]) byOctet2|=B00000010;
    if (pc->tboFct[15]) byOctet2|=B00000100;
    if (pc->tboFct[16]) byOctet2|=B00001000;
    if (pc->tboFct[17]) byOctet2|=B00010000;
    if (pc->tboFct[18]) byOctet2|=B00100000;
    if (pc->tboFct[19]) byOctet2|=B01000000;
    if (pc->tboFct[20]) byOctet2|=B10000000;
    sprintf(tcCde_pad, "<f %d %d %d>", pc->iNum, byOctet1, byOctet2);
    memcpy(tbCde_pad,tcCde_pad,strlen(tcCde_pad));
    Transmission_Serie2(&tbCde_pad[0], strlen(tcCde_pad));
  }else if (iModif<29){
    byOctet1=MASQ_21_28;
    byOctet2=B00000000;
    if (pc->tboFct[21]) byOctet2|=B00000001;
    if (pc->tboFct[22]) byOctet2|=B00000010;
    if (pc->tboFct[23]) byOctet2|=B00000100;
    if (pc->tboFct[24]) byOctet2|=B00001000;
    if (pc->tboFct[25]) byOctet2|=B00010000;
    if (pc->tboFct[26]) byOctet2|=B00100000;
    if (pc->tboFct[27]) byOctet2|=B01000000;
    if (pc->tboFct[28]) byOctet2|=B10000000;
    sprintf(tcCde_pad, "<f %d %d %d>", pc->iNum, byOctet1, byOctet2);
    memcpy(tbCde_pad,tcCde_pad,strlen(tcCde_pad));
    Transmission_Serie2(&tbCde_pad[0], strlen(tcCde_pad));
  }
}

/* codage des actions réalisées sur le joystick */
void Traitement_actions_PAD(uint8_t *ucData, int iLen){
int i;

  boModif=true;
  if (iLen==2){
    iAction_PAD=ucData[0]; 
    i=ucData[1] & 0xc0;                     // axe des Y centré = 0xa0
    if (i==0) iAction_PAD|=BT_UP;
    else if (i==0x80) iAction_PAD|=BT_DOWN;
    i=ucData[1] & 0x30;                     // axe des X centré = 0x10
    if (i==0) iAction_PAD|=BT_LEFT;
    else if (i==0x20) iAction_PAD|=BT_RIGHT;
  } 
}

/* initialisation programme */
void setup(){

  Wire.begin();                             // initialisation bus I2C 
  /* initialisation premier port série */
  //Serial.begin(115200);
  ESP32Serial.begin(115200);                    
  while (!ESP32Serial);
  ESP32Serial.flush();
  /* initialisation second port série */
  ESP32Serial2.begin(115200, SERIAL_8N1, RX2, TX2); 
  while (!ESP32Serial2);
  ESP32Serial2.flush();
  /* initialisation serveur BLE */
  if (!BLE.begin()) while (1);              // initialisation BLE
  Init_scan_BLE();
  /* initialisation OLED */
  ecran.init();                             // initialisation écran OLED
  ecran.setFont(ArialMT_Plain_10);
  ecran.flipScreenVertically();             // affichage inversé de l'écran (connecteur en haut)
  /* initialisation EEPROM */
  eeprom.begin("PAD_BLE",false);            // initialisation espace EEPROM
  Charge_MAC_EEPROM();                      // chargement adresse MAC mémorisée
  /* initialisation affichage menus */
  if (String(tcMac) != "") bySel_menu = 0;  // si PAD mémorisé = attente cnx PAD
  else bySel_menu = 2;                      // sinon apprentissage nouveau PAD
  boSel_menu_ok   = true;                   // le premier niveau de menu est validé
  bySel_menu_loco = 0;                      // position du menu des locos (après connexion PAD)
  iAction_PAD     = 0;                      // initialisation des actions du PAD
  sMessage        = "";                     // initialisation affichage message d'information
  /* initialisation pointeurs locos */
  p_loco          = NULL;                   // la liste des locos mémorisées est vide
  p_sel_loco      = NULL;                   // ptr sur sélection liste loco  
  p_selA_loco     = NULL;                   // ptr sur loco A sélectionnée  
  p_selB_loco     = NULL;                   // ptr sur loco B sélectionnée  
  p_selC_loco     = NULL;                   // ptr sur loco C sélectionnée  
  p_selD_loco     = NULL;                   // ptr sur loco D sélectionnée  
  bySel_memo      = -1;
  /* initialisation bouton suivant */
  boBt_sui        =false;
  pinMode(12, INPUT_PULLUP);
  attachInterrupt(12, bouton_suivant_ISR, FALLING);
  /* initialisation bouton enter */
  boBt_ent        =false;
  pinMode(14, INPUT_PULLUP);
  attachInterrupt(14, bouton_enter_ISR, FALLING);
}

/* boucle programme */
void loop(){
String reception;

  if (boBt_sui || boBt_ent) Action_menu_BLE();
  Affiche_menu_BLE();
  if ((boSel_menu_ok) && (bySel_menu==0)){                 // Attente connexion PAD 
    if (Lecture_BLE()){
      if (strcmp(tcMac,tcAdr_ble)==0){       
        BLE.stopScan();
        /* affichage du menu de synchronisation */
        ecran.clear();
        iPos_y=0;
        iPos_x=0;
        ecran.drawString(iPos_x, iPos_y, TITRE_MENU);
        iPos_x=ecran.getWidth()-ecran.getStringWidth("connecté");
        ecran.drawString(iPos_x, iPos_y, "connecté");
        iPos_y+=2*OFFSET_Y;
        iPos_x=(ecran.getWidth()-ecran.getStringWidth("UTILISEZ LE MODE C"))/2;
        ecran.drawString(iPos_x, iPos_y, "UTILISEZ LE MODE C");
        iPos_x=0;
        ecran.display();
        eeprom.end();                                      // fermeture de l'espace eeprom
        Charge_Locos_EEPROM();
        /* début boucle - le PAD est synchronisé */
        Gestion_PAD(peripherique);         
        /* fin boucle - le PAD n'est plus synchronisé */
        eeprom.begin("PAD_BLE", false);                    // ouverture de l'espace eeprom
        BLE.scan();                                        // démarrage scan BLE
      }
    }
  }
  /* gestion des ports série sans synchro PAD */
  int iBytes=Reception_Serie(&tbyData[0], TAILLE_BUF);
  if (iBytes>0) Transmission_Serie2(&tbyData[0], iBytes);
  int iBytes2=Reception_Serie2(&tbyData[0], TAILLE_BUF);
  if (iBytes2>0) Transmission_Serie(&tbyData[0], iBytes2);
}

/* analyse des commandes passées à la centrale et stockage des données en liste chaînée */
void Gestion_loco(char *tcCmd){
char      chaine[64], cmd[64];
int       num, reg, vit, dir, fct1, fct2;
strucLoco *pc;                                             // pointeurs courant sur structure loco
strucLoco *pp;                                             // pointeurs précédent sur structure loco

  strcpy(chaine, tcCmd);
  if (chaine[0]=='0'){                                     // arrêt génération DCC
    boGen_dcc=false;
  }else if (chaine[0]=='1'){                               // génération DCC
    boGen_dcc=true;
  }else if (chaine[0]=='t'){                               // création / mise à jour loco
    if (sscanf(chaine, "%c %d %d %d %d", cmd, &reg, &num, &vit, &dir)==5){
      pc=p_loco;                            // recherche loco
      while ((pc!=NULL) && (pc->iNum!=num)) pc=pc->ps;
      if (pc==NULL){                                       // loco non trouvée = à ajouter
        strucLoco *pn=new strucLoco;
        pn->byReg=reg;
        pn->iNum=num;
        if (vit>126) vit=126;
        pn->byVit=vit;
        if (dir==1) pn->boDir=true;
        else pn->boDir=false;
        for(int i=0;i<29;i++)pn->tboFct[i]=false;   
        pn->bySel_fct=0;            
        pn->ps=p_loco;      
        p_loco=pn;
        if (p_sel_loco==NULL) p_sel_loco=p_loco;           // init sélection affichage liste des locos
      }else{                                               // loco trouvée = à modifier
        pc->byReg=reg;
        pc->byVit=vit;
        if (dir==1) pc->boDir=true;
        else pc->boDir=false;
      }
    }
  }else if (chaine[0]=='f'){                               // fonction loco
    if (sscanf(chaine, "%c %d %d %d", cmd, &num, &fct1, &fct2)==4){
      pc=p_loco;                                           // recherche loco
      while ((pc!=NULL) && (pc->iNum!=num)) pc=pc->ps;
      if (pc!=NULL){                                       // loco trouvée
        /*  */
      }
    }else if (sscanf(chaine, "%c %d %d", cmd, &num, &fct1)==3){
      
    }
  }else if (chaine[0]=='-'){                               // suppression loco (fermeture fenêtre du régulateur JMRI)
    if (sscanf(chaine, "%c%d", cmd, &num)==2){
      pc=p_loco;                                           // recherche loco
      pp=NULL;
      while ((pc!=NULL) && (pc->iNum!=num)){
        pp=pc;
        pc=pc->ps;
      }
      if (pc!=NULL){                                       // loco trouvée = à supprimer
        if (pp==NULL) p_loco=pc->ps;                       // première loco de la liste à supprimer
        else pp->ps=pc->ps;                                // autre loco à supprimer
        delete[] pc;                                       // suppression structure loco
        p_sel_loco=p_loco;                                 // init sélection affichage liste des locos
      }
    }
  }
}

/* reconstitution de la commande */
void Reception_caracteres(byte *bMsg, int iLen){
int i, n;

  for (i=0;i<iLen;i++){
    if(char(bMsg[i])=='<'){                                // début de commande
      boDeb_cmd=true;
      sprintf(tcRec_cmd, "");
    }else if(char(bMsg[i])=='>'){                          // fin de commande
      boDeb_cmd=false;
      if ((strlen(tcRec_cmd)>0) && ((tcRec_cmd[0]=='t') || (tcRec_cmd[0]=='-') || (tcRec_cmd[0]=='0') || (tcRec_cmd[0]=='1'))){
        strcpy(tcAff_cmd, tcRec_cmd);
        Gestion_loco(tcAff_cmd);
        boModif=true;
      }
      sprintf(tcRec_cmd, "");
    }else if (boDeb_cmd){                                  // caractère en cours de réception
      n = strlen(tcRec_cmd);
      tcRec_cmd[n]=char(bMsg[i]);
      tcRec_cmd[n+1]=0;
    }
  }
}

/* connexion au PAD et traitement des actions */
void Gestion_PAD(BLEDevice appareil) {
int       x, y, i, iCount;
int       iBytes, iBytes2;
String    sCommande;
String    sMemorisation;
char      tcAff[64];
char      tcCde_pad[TAILLE_BUF];
byte      tbCde_pad[TAILLE_BUF];
boolean   bt_sel;
boolean   bt_ok;
strucLoco *pc;                                             // pointeurs courant sur structure loco


  if (!appareil.connect()) return;                         // périphérique connecté
  if (!appareil.discoverService("1812")){                  // découverte des services possible
    appareil.disconnect();
    return;
  }
  BLEService hidService = appareil.service("1812");        // liste des services
  iCount=hidService.characteristicCount();                 // nombre de services
  for (i=0;i<iCount;i++){                                  // parcours des services
    BLECharacteristic bc=hidService.characteristic(i);
    if (strcasecmp(bc.uuid(),"2a4d")==0){                  // la notification est active
      bc.subscribe();                                      // inscription au service
      bc.setEventHandler(BLEWritten, HIDReportWritten);    // synchronisation avec les évènements du PAD
    }
  }
  delay(2000);                                             // délais nécessaire à l'initialisation de la connexion
  while (appareil.connected()){                            // tant que le périphérique est connecté
    
    if (boModif){
      if(millis() - lOld_millis_PAD >= 30){
        lOld_millis_PAD = millis();
        boModif=false;
        /* efface zone écran dédié traitement actions PAD */
        for (x=0;x<=128;x++){
          for (y=iPos_y;y<=64;y++){
            ecran.clearPixel(x, y);
          }
        }
        /*  */
        sCommande="";
        bt_ok=false;
        bt_sel=false;
        if       (iAction_PAD==    0){                     // Aucune commande - tous les boutons et joystick au repos
          lOld_millis_PAD=0;
          lOld_millis_FCT=0;
        }else if (iAction_PAD==    1){                     // sCommande="Bouton C";
          if (bySel_menu_loco==1){ // emplacement loco mémorisée
            bySel_memo=2;
            p_sel_loco=p_selC_loco;
          }
        }else if (iAction_PAD==   2){                      // sCommande="Bouton A";
          if (bySel_menu_loco==1){ // emplacement loco mémorisée
            bySel_memo=0;
            p_sel_loco=p_selA_loco;
          }
        }else if (iAction_PAD==   8){                      // sCommande="Bouton B";
          if (bySel_menu_loco==1){ // emplacement loco mémorisée
            bySel_memo=1;
            p_sel_loco=p_selB_loco;
          }
        }else if (iAction_PAD==  16){                      // sCommande="Bouton D";
          if (bySel_menu_loco==1){ // emplacement loco mémorisée
            bySel_memo=3;
            p_sel_loco=p_selD_loco;
          }
        }else if (iAction_PAD==  64){                      // sCommande="Bouton Ok";     
          bt_ok=true;
          if (bySel_menu_loco==1){                         // emplacement loco mémorisée
            if (p_sel_loco!=NULL){                         // une loco est mémorisée
              p_sel_loco->tboFct[p_sel_loco->bySel_fct]=!p_sel_loco->tboFct[p_sel_loco->bySel_fct];
              Envoyer_commande_fonctions(p_sel_loco,p_sel_loco->bySel_fct);
            }
          }
        }else if (iAction_PAD==  65){                      // sCommande="Bouton Ok + loco C";     
          bt_ok=true;
          if ((bySel_menu_loco==1) && (p_selC_loco!=NULL)){ // emplacement loco mémorisée
            p_selC_loco->byVit=0;                        // vitesse à zéro
            Envoyer_commande_loco(p_selC_loco);          // envoyer commande
          }
          iAction_PAD=0;                               // pas de répétition de la commande
        }else if (iAction_PAD==  66){                      // sCommande="Bouton Ok + loco A";     
          bt_ok=true;
          if ((bySel_menu_loco==1) && (p_selA_loco!=NULL)){ // emplacement loco mémorisée
            p_selA_loco->byVit=0;                        // vitesse à zéro
            Envoyer_commande_loco(p_selA_loco);          // envoyer commande
          }
          iAction_PAD=0;                               // pas de répétition de la commande
        }else if (iAction_PAD==  72){                      // sCommande="Bouton Ok + loco B";     
          bt_ok=true;
          if ((bySel_menu_loco==1) && (p_selB_loco!=NULL)){ // emplacement loco mémorisée
            p_selB_loco->byVit=0;                        // vitesse à zéro
            Envoyer_commande_loco(p_selB_loco);          // envoyer commande
          }
          iAction_PAD=0;                               // pas de répétition de la commande
        }else if (iAction_PAD==  80){                      // sCommande="Bouton Ok + loco D";     
          bt_ok=true;
          if ((bySel_menu_loco==1) && (p_selD_loco!=NULL)){ // emplacement loco mémorisée
            p_selD_loco->byVit=0;                        // vitesse à zéro
            Envoyer_commande_loco(p_selD_loco);          // envoyer commande
          }
          iAction_PAD=0;                               // pas de répétition de la commande
        }else if (iAction_PAD== 128){                      // sCommande="Sélection";
          bt_sel=true;
        }else if (iAction_PAD== 256){                      // sCommande="Joystick Gauche";     
          if (bySel_menu_loco==1){                         // locos mémorisée
            if (p_sel_loco!=NULL){                         // défilement fonctions loco vers la gauche
              if(millis() - lOld_millis_FCT >= 250){
                lOld_millis_FCT = millis();
                if (p_sel_loco->bySel_fct>0){
                  p_sel_loco->bySel_fct-=1;
                }else p_sel_loco->bySel_fct=28;
              }
              boModif=true;
            }
          }  
        }else if (iAction_PAD== 512){                      // sCommande="Joystick Droit";  
          if (bySel_menu_loco==0){                         // liste des locos disponibles 
            if (p_sel_loco!=NULL) {
              p_sel_loco=p_sel_loco->ps;
              if (p_sel_loco==NULL) p_sel_loco=p_loco;
            }
          }else if (bySel_menu_loco==1){                   // locos mémorisée  
            if (p_sel_loco!=NULL){                         // défilement fonctions loco vers la droite
              if(millis() - lOld_millis_FCT >= 250){
                lOld_millis_FCT = millis();
                if (p_sel_loco->bySel_fct<28){
                  p_sel_loco->bySel_fct+=1;
                }else p_sel_loco->bySel_fct=0;
              }
              boModif=true;
            }
          }
        }else if (iAction_PAD==1024){                      // sCommande="Joystick Haut";    
          if (bySel_menu_loco==1){                         // emplacement loco mémorisée
            if (p_sel_loco!=NULL){                         // augmentation vitesse loco
              p_sel_loco->byVit++;
              if (p_sel_loco->byVit>126) p_sel_loco->byVit=126; 
              Envoyer_commande_loco(p_sel_loco);
              boModif=true;  
            }
          }
        }else if (iAction_PAD==2048){                      // sCommande="Joystick Bas";      
          if (bySel_menu_loco==1){                         // emplacement loco mémorisée
            if (p_sel_loco!=NULL){                         // diminution vitesse loco
              p_sel_loco->byVit--;
              if (p_sel_loco->byVit>126) p_sel_loco->byVit=0;   
              Envoyer_commande_loco(p_sel_loco);
              boModif=true; 
            }
          }
        }else if (iAction_PAD==2304){                      // sCommande="Joystick Bas Gauche";      
        }else if (iAction_PAD==2560){                      // sCommande="Joystick Bas Droit";      
        }else if (iAction_PAD==1280){                      // sCommande="Joystick Haut Gauche";       
        }else if (iAction_PAD==1536){                      // sCommande="Joystick Haut Droit";      
        }else if (iAction_PAD== 130){                      // sCommande="Retour + loco A";
          if (bySel_menu_loco==0){                         // depuis liste des locos disponibles 
            if (p_sel_loco!=NULL){
              p_selA_loco=p_sel_loco;
              bySel_menu_loco=1;
              bySel_memo=0;
            }
          }else if (bySel_menu_loco==1){                   // depuis emplacement loco mémorisé
            if (p_selA_loco!=NULL){                        // une loco est mémorisée en A
              p_selA_loco->boDir=!p_selA_loco->boDir;      // changement direction
              p_selA_loco->byVit=0;                        // vitesse à zéro
              Envoyer_commande_loco(p_selA_loco);          // envoyer commande
              iAction_PAD=0;                               // pas de répétition de la commande
            }
          }
        }else if (iAction_PAD== 136){                      // sCommande="Retour + loco B";
          if (bySel_menu_loco==0){                         // liste des locos disponibles 
            if (p_sel_loco!=NULL){
              p_selB_loco=p_sel_loco;
              bySel_menu_loco=1;
              bySel_memo=1;
            }
          }else if (bySel_menu_loco==1){                   // emplacement loco mémorisée
            if (p_selB_loco!=NULL){                        // une loco est mémorisée en B
              p_selB_loco->boDir=!p_selB_loco->boDir;      // changement direction
              p_selB_loco->byVit=0;                        // vitesse à zéro   
              Envoyer_commande_loco(p_selB_loco);          // envoyer commande
              iAction_PAD=0;                               // pas de répétition de la commande
            }
          }
        }else if (iAction_PAD== 129){                      // sCommande="Retour + loco C";
          if (bySel_menu_loco==0){                         // liste des locos disponibles 
            if (p_sel_loco!=NULL){
              p_selC_loco=p_sel_loco;
              bySel_menu_loco=1;
              bySel_memo=2;
            }
          }else if (bySel_menu_loco==1){                   // emplacement loco mémorisée
            if (p_selC_loco!=NULL){                        // une loco est mémorisée en C
              p_selC_loco->boDir=!p_selC_loco->boDir;      // changement direction
              p_selC_loco->byVit=0;                        // vitesse à zéro
              Envoyer_commande_loco(p_selC_loco);          // envoyer commande
              iAction_PAD=0;                               // pas de répétition de la commande
            }
          }
        }else if (iAction_PAD== 144){                      // sCommande="Retour + loco D";
          if (bySel_menu_loco==0){                         // liste des locos disponibles 
            if (p_sel_loco!=NULL){
              p_selD_loco=p_sel_loco;
              bySel_menu_loco=1;
              bySel_memo=3;
            }
          }else if (bySel_menu_loco==1){                   // emplacement loco mémorisée
            if (p_selD_loco!=NULL){                        // une loco est mémorisée en D
              p_selD_loco->boDir=!p_selD_loco->boDir;      // changement direction
              p_selD_loco->byVit=0;                        // vitesse à zéro
              Envoyer_commande_loco(p_selD_loco);          // envoyer commande
              iAction_PAD=0;                               // pas de répétition de la commande
            }
          }
        }else if (iAction_PAD==1152){                      // sCommande="Sélection Menu choix loco"; 
          bt_sel=true;      
          bySel_menu_loco=0;
          p_sel_loco=p_loco;
        }else if (iAction_PAD==2176){                      // sCommande="Sélection Menu loco mémo";
          bt_sel=true;
          bySel_menu_loco=1;
        }else if (iAction_PAD==  24){                      // sCommande="Arrêt / démarrage : A + B + C + D";
          boGen_dcc=!boGen_dcc;
          if (boGen_dcc) sprintf(tcCde_pad,"<1>");
          else sprintf(tcCde_pad,"<0>");
          memcpy(tbCde_pad,tcCde_pad,strlen(tcCde_pad));
          Transmission_Serie2(&tbCde_pad[0], strlen(tcCde_pad));
          pc=p_loco;                                       // parcours toutes locos
          while (pc!=NULL){
            for(int i=0;i<29;i++)pc->tboFct[i]=false;      // RàZ des fonctions
            if (boGen_dcc && (pc->byVit>0)){               // RàZ des vitesses si allumage et vitesse déjà > 0
              pc->byVit=0;
              Envoyer_commande_loco(pc);
            }
            pc->bySel_fct=0;                               // RàZ de la fonction sélectionnée 
            pc=pc->ps;
          }       
          iAction_PAD=0;                                   // pas de répétition de la commande
        }else if (iAction_PAD== 514){                      // sCommande="Loco A + joystick droit";
          if (bySel_menu_loco==1){                         // depuis emplacement loco mémorisé
            if (p_selA_loco!=NULL){                        // une loco est mémorisée en A
              eeprom.begin("LOCO_BLE", false);             // ouverture de l'espace eeprom LOCO
              eeprom.putUInt("ADDR_A", p_selA_loco->iNum);
              sMemorisation="Loco A enregistrée : "+String(p_selA_loco->iNum);
              eeprom.end();
            }
          }  
          iAction_PAD=0;                                   // pas de répétition de la commande
        }else if (iAction_PAD== 258){                      // sCommande="Loco A + joystick gauche";
          if (bySel_menu_loco==1){                         // depuis emplacement loco mémorisé
            eeprom.begin("LOCO_BLE", false);               // ouverture de l'espace eeprom LOCO
            eeprom.remove("ADDR_A");
            sMemorisation="Enreg. Loco A supprimé";
            eeprom.end();
          }
          iAction_PAD=0;                                   // pas de répétition de la commande
        }else if (iAction_PAD== 520){                      // sCommande="Loco B + joystick droit";
          if (bySel_menu_loco==1){                         // depuis emplacement loco mémorisé
            if (p_selB_loco!=NULL){                        // une loco est mémorisée en B
              eeprom.begin("LOCO_BLE", false);             // ouverture de l'espace eeprom LOCO
              eeprom.putUInt("ADDR_B", p_selB_loco->iNum);
              sMemorisation="Loco B enregistrée : "+String(p_selB_loco->iNum);
              eeprom.end();
            }
          }  
          iAction_PAD=0;                                   // pas de répétition de la commande
        }else if (iAction_PAD== 264){                      // sCommande="Loco B + joystick gauche";
          if (bySel_menu_loco==1){                         // depuis emplacement loco mémorisé
            eeprom.begin("LOCO_BLE", false);               // ouverture de l'espace eeprom LOCO
            eeprom.remove("ADDR_B");
            sMemorisation="Enreg. Loco B supprimé";
            eeprom.end();
          }
          iAction_PAD=0;                                   // pas de répétition de la commande
        }else if (iAction_PAD== 513){                      // sCommande="Loco C + joystick droit";
          if (bySel_menu_loco==1){                         // depuis emplacement loco mémorisé
            if (p_selC_loco!=NULL){                        // une loco est mémorisée en C
              eeprom.begin("LOCO_BLE", false);             // ouverture de l'espace eeprom LOCO
              eeprom.putUInt("ADDR_C", p_selC_loco->iNum);
              sMemorisation="Loco C enregistrée : "+String(p_selC_loco->iNum);
              eeprom.end();
            }
          }  
          iAction_PAD=0;                                   // pas de répétition de la commande
        }else if (iAction_PAD== 257){                      // sCommande="Loco C + joystick gauche";
          if (bySel_menu_loco==1){                         // depuis emplacement loco mémorisé
            eeprom.begin("LOCO_BLE", false);               // ouverture de l'espace eeprom LOCO
            eeprom.remove("ADDR_C");
            sMemorisation="Enreg. Loco C supprimé";
            eeprom.end();
          }
          iAction_PAD=0;                                   // pas de répétition de la commande
        }else if (iAction_PAD== 528){                      // sCommande="Loco D + joystick droit";
          if (bySel_menu_loco==1){                         // depuis emplacement loco mémorisé
            if (p_selD_loco!=NULL){                        // une loco est mémorisée en D
              eeprom.begin("LOCO_BLE", false);             // ouverture de l'espace eeprom LOCO
              eeprom.putUInt("ADDR_D", p_selD_loco->iNum);
              sMemorisation="Loco D enregistrée : "+String(p_selD_loco->iNum);
              eeprom.end();
            }
          }  
          iAction_PAD=0;                                   // pas de répétition de la commande
        }else if (iAction_PAD== 272){                      // sCommande="Loco D + joystick gauche";
          if (bySel_menu_loco==1){                         // depuis emplacement loco mémorisé
            eeprom.begin("LOCO_BLE", false);               // ouverture de l'espace eeprom LOCO
            eeprom.remove("ADDR_D");
            sMemorisation="Enreg. Loco D supprimé";
            eeprom.end();
          }
          iAction_PAD=0;                                   // pas de répétition de la commande
        }else if (iAction_PAD!=   0) sCommande=String(iAction_PAD);
        
        iPos_x=0;
        if (sCommande!="") ecran.drawString(iPos_x, iPos_y, sCommande);    
        else if (sMemorisation!=""){
          ecran.drawString(iPos_x, iPos_y, sMemorisation);  
          sMemorisation="";  
        }else{
          if (bySel_menu_loco==0){                         // liste des locos mémorisées
            if (bt_sel) ecran.drawString(iPos_x, iPos_y, "> Mémorisation ABCD ?");
            else ecran.drawString(iPos_x, iPos_y, MENU_LOCO[bySel_menu_loco]);
            // affichage loco sélectionnée
            if (p_sel_loco==NULL) ecran.drawString(iPos_x, iPos_y+OFFSET_Y, "Aucune loco dans la liste");
            else ecran.drawString(iPos_x, iPos_y+OFFSET_Y, "   Adresse : "+String(p_sel_loco->iNum));
          }else if (bySel_menu_loco==1){                   // état des 4 emplacements de locos mémorisables
            p_sel_loco=NULL;
            if (bySel_memo==0){                            // emplacement A
              if (p_selA_loco==NULL){
                ecran.drawString(iPos_x, iPos_y, "LOCO A");
                ecran.drawString(iPos_x, iPos_y+OFFSET_Y, MENU_NDEF);
              }else{
                p_sel_loco=p_selA_loco;
                ecran.drawString(iPos_x, iPos_y, "LOCO A - id "+String(p_sel_loco->iNum));
              }
            }else if (bySel_memo==1){                      // emplacement B
              if (p_selB_loco==NULL){
                ecran.drawString(iPos_x, iPos_y, "LOCO B");
                ecran.drawString(iPos_x, iPos_y+OFFSET_Y, MENU_NDEF);
              }else{
                p_sel_loco=p_selB_loco;
                ecran.drawString(iPos_x, iPos_y, "LOCO B - id "+String(p_sel_loco->iNum));
              }
            }else if (bySel_memo==2){                      // emplacement C
              if (p_selC_loco==NULL){
                ecran.drawString(iPos_x, iPos_y, "LOCO C");
                ecran.drawString(iPos_x, iPos_y+OFFSET_Y, MENU_NDEF);
              }else{
                p_sel_loco=p_selC_loco;
                ecran.drawString(iPos_x, iPos_y, "LOCO C - id "+String(p_sel_loco->iNum));
              }
            }else if (bySel_memo==3){                      // emplacement D
              if (p_selD_loco==NULL){
                ecran.drawString(iPos_x, iPos_y, "LOCO D");
                ecran.drawString(iPos_x, iPos_y+OFFSET_Y, MENU_NDEF);
              }else{
                p_sel_loco=p_selD_loco;
                ecran.drawString(iPos_x, iPos_y, "LOCO D - id "+String(p_sel_loco->iNum));
              }
            }else{                                         // Aucun emplacement
              ecran.drawString(iPos_x, iPos_y, MENU_NSEL);           
            }
            /* affichage des informations de la loco sélectionnée */
            if (p_sel_loco!=NULL){
              if (p_sel_loco->boDir) ecran.drawString(iPos_x, iPos_y+OFFSET_Y, "-> Marche avant");
              else ecran.drawString(iPos_x, iPos_y+OFFSET_Y, "<- Marche arrière");
              ecran.setFont(ArialMT_Plain_16);
              iPos_x=ecran.getWidth()-ecran.getStringWidth(String(p_sel_loco->byVit));
              ecran.drawString(iPos_x, iPos_y, String(p_sel_loco->byVit));
              ecran.setFont(ArialMT_Plain_10);
              /* affichage des fonctions de 0 à 28 visibles à l'écran */
              for(int i=0;i<8;i++){
                if ((p_sel_loco->bySel_fct-3+i>=0) && (p_sel_loco->bySel_fct-3+i<29)){
                  ecran.drawString(i*20, iPos_y+5+(2*OFFSET_Y), "f"+String(p_sel_loco->bySel_fct-3+i));
                  /* affichage fonction si active */
                  if (p_sel_loco->tboFct[p_sel_loco->bySel_fct-3+i]){
                    int iLarg=ecran.getStringWidth("f"+String(p_sel_loco->bySel_fct-3+i));
                    ecran.fillRect(i*20, iPos_y+6+(3*OFFSET_Y), iLarg, 3);
                  }
                }
              }
              /* affichage flèche de sélection */
              ecran.fillTriangle(62, iPos_y+5+(4*OFFSET_Y), 65, iPos_y+9+(3*OFFSET_Y), 68, iPos_y+5+(4*OFFSET_Y));
            }
          }
        }
        ecran.display();
      }
    }
    /* gestion des ports série pendant synchro PAD */
    iBytes=Reception_Serie(&tbyData[0], TAILLE_BUF);
    if (iBytes>0) Transmission_Serie2(&tbyData[0], iBytes);
    iBytes2=Reception_Serie2(&tbyData[0], TAILLE_BUF);
    if (iBytes2>0) Transmission_Serie(&tbyData[0], iBytes2);
  }
  /* PAD déconnecté */
}
